const express = require('express');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { body, validationResult } = require('express-validator');
const { mysqlHelper } = require('../database/mysql-helper');
const { authenticateToken, requireAdmin, validateUserExists } = require('../middleware/auth');

const router = express.Router();

// Validation rules
const registerValidation = [
  body('email').isEmail().normalizeEmail().withMessage('Valid email is required'),
  body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters')
];

const adminRegisterValidation = [
  body('name').trim().isLength({ min: 2 }).withMessage('Name must be at least 2 characters'),
  body('email').isEmail().normalizeEmail().withMessage('Valid email is required'),
  body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
  body('phone').optional().trim().isLength({ min: 10 }).withMessage('Valid phone number required')
];

const loginValidation = [
  body('email').trim().notEmpty().withMessage('Username/Email is required'),
  body('password').notEmpty().withMessage('Password is required')
];

// Helper function to generate JWT
const generateToken = (user) => {
  return jwt.sign(
    { 
      id: user.id, 
      email: user.email, 
      role: user.role 
    },
    process.env.JWT_SECRET,
    { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
  );
};

// Register new user (public - minimal details)
router.post('/register', registerValidation, async (req, res) => {
  try {
    // Check validation errors
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { email, password } = req.body;

    // Check if user already exists
    const existingUser = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT id FROM users WHERE email = ?', [email], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (existingUser) {
      return res.status(409).json({ error: 'Email already registered' });
    }

    // Hash password
    const saltRounds = parseInt(process.env.BCRYPT_ROUNDS) || 12;
    const hashedPassword = await bcrypt.hash(password, saltRounds);

    // Insert new user (default name from email, client role, active status)
    const name = email.split('@')[0]; // Use email prefix as default name
    const result = await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'INSERT INTO users (name, email, password, phone, role, status) VALUES (?, ?, ?, ?, ?, ?)',
        [name, email, hashedPassword, '', 'client', 'active'],
        (err, result) => {
          if (err) reject(err);
          else resolve({ id: result.lastID });
        }
      );
    });

    // Get created user
    const newUser = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT id, name, email, phone, role, status, created_at FROM users WHERE id = ?',
        [result.id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    // Generate token
    const token = generateToken(newUser);

    res.status(201).json({
      message: 'User registered successfully',
      token,
      user: newUser
    });

  } catch (error) {
    console.error('Registration error:', error);
    res.status(500).json({ error: 'Server error during registration' });
  }
});

// Register new admin user (admin only - full details)
router.post('/admin/register', authenticateToken, requireAdmin, adminRegisterValidation, async (req, res) => {
  try {
    // Check validation errors
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, email, password, phone, role = 'client' } = req.body;

    // Check if user already exists
    const existingUser = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT id FROM users WHERE email = ?', [email], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (existingUser) {
      return res.status(409).json({ error: 'Email already registered' });
    }

    // Hash password
    const saltRounds = parseInt(process.env.BCRYPT_ROUNDS) || 12;
    const hashedPassword = await bcrypt.hash(password, saltRounds);

    // Validate role
    const validRole = ['admin', 'client'].includes(role) ? role : 'client';

    // Insert new user
    const result = await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'INSERT INTO users (name, email, password, phone, role, status) VALUES (?, ?, ?, ?, ?, ?)',
        [name, email, hashedPassword, phone || '', validRole, 'active'],
        (err, result) => {
          if (err) reject(err);
          else resolve({ id: result.lastID });
        }
      );
    });

    // Get created user
    const newUser = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT id, name, email, role, phone, status, created_at FROM users WHERE id = ?',
        [result.id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    res.status(201).json({
      message: 'User created successfully',
      user: newUser
    });

  } catch (error) {
    console.error('Admin registration error:', error);
    res.status(500).json({ error: 'Server error during user creation' });
  }
});

// Login user
router.post('/login', loginValidation, async (req, res) => {
  try {
    // Check validation errors
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { email, password } = req.body;

    // Find user by email
    const user = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT id, name, email, password, role, status, avatar, created_at FROM users WHERE email = ?',
        [email],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    if (!user) {
      return res.status(401).json({ error: 'Invalid email or password' });
    }

    // Check if user is active
    if (user.status !== 'active') {
      return res.status(401).json({ error: 'Account is inactive. Please contact administrator.' });
    }

    // Verify password
    const isValidPassword = await bcrypt.compare(password, user.password);
    if (!isValidPassword) {
      return res.status(401).json({ error: 'Invalid email or password' });
    }

    // Remove password from response
    const { password: _, ...userWithoutPassword } = user;

    // Generate token
    const token = generateToken(user);

    res.json({
      message: 'Login successful',
      token,
      user: userWithoutPassword
    });

  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Server error during login' });
  }
});

// Verify token and get current user
router.get('/me', authenticateToken, validateUserExists, (req, res) => {
  res.json({
    user: req.userDetails
  });
});

// Refresh token
router.post('/refresh', authenticateToken, validateUserExists, (req, res) => {
  const newToken = generateToken(req.userDetails);
  
  res.json({
    message: 'Token refreshed successfully',
    token: newToken,
    user: req.userDetails
  });
});

// Logout (client-side token removal, no server-side action needed for JWT)
router.post('/logout', authenticateToken, (req, res) => {
  res.json({ message: 'Logout successful' });
});

module.exports = router;
